<?php
/**
 * Plugin Name: Tickera - Predefined Ticket Codes
 * Plugin URI: https://tickera.com
 * Description: Unofficial add-on: Allow the merchant to import pre-defined ticket codes via CSV. Option can be found at Tickera > Settings.
 * Author: Tickera.com
 * Author URI: http://tickera.com/
 * Version: 1.2
 * Copyright 2017 Tickera (http://tickera.com/)
 */

if ( ! defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

if ( ! class_exists( 'TC_Predefined_Ticket_Codes' ) ) {

    class TC_Predefined_Ticket_Codes {

        var $version = '1.2';
        var $title = 'Tickera - Predefined Ticket Codes';
        var $name = 'tc-predefined-ticket-codes';
        var $dir_name = 'tc-predefined-ticket-codes';
        var $location = 'plugins';
        var $plugin_dir = '';
        var $plugin_url = '';

        function __construct() {

            self::init_variables();
            self::init_classes();

            $setting = get_option( 'tc_predefined_ticket_codes_setting' );
            $enabled = isset( $setting[ 'enabled' ] ) ? $setting[ 'enabled' ] : 'no';

            add_action( 'activated_plugin', array( $this, 'init' ), 99 );
            add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

            // Additional section in settings page
            add_filter( 'tc_settings_new_menus', array( $this, 'additional_predefined_ticket_codes_setting' ) );
            add_action( 'tc_settings_menu_tickera_predefined_ticket_codes', array( $this, 'additional_predefined_ticket_codes_setting_content' ) );

            // Process import
            add_action( 'admin_post_tickera_import_predefined_ticket_codes', array( $this, 'process_import' ) );

            // Save Settings
            add_action( 'admin_post_tickera_save_predefined_ticket_codes', array( $this, 'save_settings' ) );

            // Ajax Calls, Update and Delete Ticket Codes
            add_action( 'wp_ajax_tickera_predefined_ticket_code_delete_action', array( $this, 'delete_ticket_code_ajax' ) );
            add_action( 'wp_ajax_tickera_predefined_ticket_code_bulk_delete_action', array( $this, 'bulk_delete_ticket_code_ajax' ) );
            add_action( 'wp_ajax_tickera_predefined_ticket_code_update_action', array( $this, 'update_ticket_code_ajax' ) );
            add_action( 'wp_ajax_tickera_predefined_ticket_code_sync_action', array( $this, 'sync_ticket_code_ajax' ) );

            // Ajax Calls. Sortable column
            add_action( 'wp_ajax_tickera_predefined_ticket_code_sort_action', array( $this, 'sort_table' ) );

            // Set Pre-defined Ticket Code to available once ticket is permanently deleted.
            add_action( 'before_delete_post', array( $this, 'make_ticket_code_available' ), 10, 1 );

            // Sync pre-defined ticket codes availability
            add_action( 'tcptc_after_import_process_completed', array( $this, 'sync_ticket_codes_availability' ) );

            if ( 'yes' == $enabled ) {

                $stop_sales = isset( $setting[ 'stop_sales' ] ) ? $setting[ 'stop_sales' ] : 'no';

                /**
                 * Stop Ticket Sales
                 */
                if ( 'yes' == $stop_sales ) {
                    add_filter( 'tc_is_ticket_type_sales_available', array( $this, 'set_ticket_types_unavailable' ), 99, 2 );
                    add_filter( 'tc_cart_error_number', array( $this, 'additional_cart_errors' ), 99 );

                    // Bridge for Woocommerce
                    if ( true == apply_filters( 'tc_is_woo', false ) ) {
                        add_action( 'woocommerce_check_cart_items', array( $this, 'additional_woo_cart_errors' ) );
                    }
                }

                // Modify Ticket Code
                add_filter( 'tc_ticket_code', array( $this, 'modify_ticket_code' ), 99, 2 );
            }
        }

        function init() {

            $plugin_slug = basename( __DIR__ ) . '/' . basename( __FILE__ );
            $active_plugins = get_option( 'active_plugins', [] );
            $index = array_search( $plugin_slug, $active_plugins );

            if ( $index !== false ) {
                unset( $active_plugins[ $index ] );
                $active_plugins[] = $plugin_slug;
                update_option( 'active_plugins', array_values( $active_plugins ) );
            }

            /**
             * Create required database table.
             */
            $db = new TC_PTC_DB();
            $db->create_tables();
        }

        function init_variables() {

            if ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . $this->dir_name . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'subfolder-plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/' . $this->dir_name . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );

            } elseif ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );

            } elseif ( is_multisite() && defined( 'WPMU_PLUGIN_URL' ) && defined( 'WPMU_PLUGIN_DIR' ) && file_exists( WPMU_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'mu-plugins';
                $this->plugin_dir = WPMU_PLUGIN_DIR;
                $this->plugin_url = WPMU_PLUGIN_URL;
            } else {
                wp_die( sprintf( __( 'There was an issue determining where %s is installed. Please reinstall it.', 'tc' ), $this->title ) );
            }
        }

        function init_classes() {
            require_once $this->plugin_dir . 'includes/classes/class.db.php';
            require_once $this->plugin_dir . 'includes/classes/class.session.php';
            require_once $this->plugin_dir . 'includes/classes/class.setting.php';
        }

        /**
         * Load Assets
         */
        function admin_enqueue_scripts() {
            wp_enqueue_style( $this->name . '-admin', plugin_dir_url( __FILE__ ) . 'assets/css/admin.css', array(), $this->version );
            wp_enqueue_script( $this->name . '-admin', plugin_dir_url( __FILE__ ) . 'assets/js/admin.js', array( 'jquery' ), $this->version, true );
            wp_localize_script( $this->name . '-admin', 'tc_ptc', array(
                    'ajaxUrl' => apply_filters( 'tc_ajaxurl', admin_url( 'admin-ajax.php', ( is_ssl() ? 'https' : 'http' ) ) ),
                    'trashConfirmationMessage' => __( 'Are you sure to remove this code?', 'tc' ),
                    'queryFailed' => __( 'An error occured during database query. Please try again.', 'tc' ),
                    'used' => __( 'used', 'tc' ),
                    'available' => __( 'available', 'tc' ),
                    'save' => __( 'save', 'tc' ),
                    'cancel' => __( 'cancel', 'tc' ),
                    'confirm_bulk_delete_message' => __( 'Action is irreversible. Please confirm that you would like to remove all existing predefined ticket codes.', 'tc' )
                )
            );
        }

        /**
         * Additional Predefined Ticket Codes Menu in Tickera > Settings
         *
         * @param $settings_tabs
         * @return mixed
         */
        function additional_predefined_ticket_codes_setting( $settings_tabs ) {
            $settings_tabs[ 'tickera_predefined_ticket_codes' ] = __( 'Predefined Ticket Codes', 'tc' );
            return $settings_tabs;
        }

        /**
         * Predefined Ticket Codes Content
         */
        function additional_predefined_ticket_codes_setting_content() {
            require_once $this->plugin_dir . 'includes/admin-pages/settings.php';
        }

        /**
         * Start importing ticket codes
         */
        function process_import() {

            // Clear session/error
            $session = new TC_PTC_Session();
            $session->destroy();

            @error_reporting( E_ERROR );
            @set_time_limit( 0 );
            @ini_set( 'max_input_time', 3600 * 3 );
            @ini_set( 'max_execution_time', 3600 * 3 );

            if ( $_FILES[ 'tc_import_predefined_ticket_codes_file' ][ 'size' ] > 0 && 'text/csv' == stripslashes( $_FILES[ 'tc_import_predefined_ticket_codes_file' ][ 'type' ] ) ) {

                // Get the csv file
                $temp_file = $_FILES[ 'tc_import_predefined_ticket_codes_file' ][ 'tmp_name' ];
                $csv = fopen( $temp_file, 'r' );

                $row = 0;
                $ticket_codes = [];
                while ( ! feof( $csv ) ) {

                    $data = fgetcsv( $csv, 100000, ",", "'" );

                    if ( $row && $data ) {
                        $ticket_code = sanitize_text_field( $data[0] );
                        $ticket_codes[] = [ $ticket_code ];
                    }

                    $row++;
                }

                $db = new TC_PTC_DB();
                $result = $db->insert( $ticket_codes );

                if ( is_array( $result ) && isset( $result[ 'error' ] ) ) {
                    $session->set( $result[ 'error' ], 'error' );

                } else {
                    $session->set( 'A total of ' . ( $row - 1 ) . ' ticket codes have been successfully added.', 'notice' );
                    do_action( 'tcptc_after_import_process_completed', $ticket_codes );
                }
            }

            self::return_to_settings_page();
            exit;
        }

        /**
         * Use first those available pre-defined ticket codes.
         *
         * @param $ticket_code
         * @param $ticket_type_id
         * @return mixed
         */
        function modify_ticket_code( $ticket_code, $ticket_type_id ) {

            $db = new TC_PTC_DB();
            $result = $db->select( [ 'status' => 1 ], 1 );

            if ( $result && ! isset( $result[ 'error' ] ) ) {
                $result = reset( $result );
                $id = $result->id;
                $predefined_ticket_code = $result->ticket_code;

                if ( self::validate_ticket_code( $predefined_ticket_code ) ) {
                    $db->update( [ 'status' => 0 ], [ 'id' => $id ] );
                    $ticket_code = $predefined_ticket_code;
                }
            }

            return $ticket_code;
        }

        /**
         * Update existing pre-defined existing ticket code.
         * Not applicable to tickets that were already used.
         */
        function update_ticket_code_ajax() {
            $data = isset( $_POST[ 'postData' ] ) ? tc_sanitize_array2( $_POST[ 'postData' ] ) : [];
            $ticket_code_id = ( isset( $data[ 'ticket_code_id' ] ) ) ? (int) $data[ 'ticket_code_id' ] : '';
            $ticket_code = ( isset( $data[ 'ticket_code' ] ) ) ? sanitize_text_field( $data [ 'ticket_code' ] ) : '';

            if ( $ticket_code_id && $ticket_code ) {
                $db = new TC_PTC_DB();
                $result = $db->update( [ 'ticket_code' => $ticket_code ], [ 'id' => $ticket_code_id, 'status' => 1 ] );
                wp_send_json( $result );
            }
        }

        /**
         * Delete existing pre-defined ticket code
         */
        function delete_ticket_code_ajax() {
            $data = isset( $_POST[ 'postData' ] ) ? tc_sanitize_array2( $_POST[ 'postData' ] ) : [];
            if ( isset( $data[ 'ticket_code_id' ] ) && $data[ 'ticket_code_id' ] ) {
                $db = new TC_PTC_DB();
                $result = $db->delete( [ 'id' => (int) $data[ 'ticket_code_id' ], 'status' => 1 ] );
                wp_send_json( $result );
            }
        }

        /**
         * Bulk Delete existing pre-defined ticket code
         */
        function bulk_delete_ticket_code_ajax() {
            $db = new TC_PTC_DB();
            $result = $db->truncate();
            wp_send_json( $result );
        }

        /**
         * Mark predefined ticket codes available if the associated ticket instance is deleted.
         *
         * @param $post_id
         */
        function make_ticket_code_available( $post_id ) {

            if ( 'tc_tickets_instances' == get_post_type( $post_id ) ) {

                $ticket_code = get_post_meta( $post_id, 'ticket_code', true );

                if ( $ticket_code ) {
                    $db = new TC_PTC_DB();
                    $db->update( [ 'status' => 1 ], [ 'ticket_code' => $ticket_code ] );
                }
            }
        }

        function save_settings() {

            if ( isset( $_POST ) && is_array( $_POST ) ) {

                $setting = [];

                foreach ( $_POST as $key => $value ) {
                    if ( preg_match( '/_option/', $key ) ) {
                        $key = str_replace( '_option', '', $key );
                        $setting[ $key ] = sanitize_text_field( $value );
                    }
                }

                update_option( 'tc_predefined_ticket_codes_setting', $setting );
            }

            self::return_to_settings_page();
        }

        /**
         * Sync and check ticket codes availability
         */
        function sync_ticket_code_ajax() {

            $complete = false; $success = true; $error = ''; $ticket_codes_availability = [];

            $data = isset( $_POST[ 'postData' ] ) ? tickera_sanitize_array( $_POST[ 'postData' ] ) : [];
            $page = isset( $data[ 'page' ] ) ? (int) $data[ 'page' ] : 1;

            $db = new TC_PTC_DB();
            $results = $db->select( [], 20, $page );

            if ( isset( $results[ 'error' ] ) ) {
                $success = false; $complete = true; $error = $results[ 'error' ];

            } elseif ( ! $results ) {
                $complete = true;

            } else {

                $ticket_codes = [];
                foreach ( $results as $result ) {
                    $ticket_codes[] = [ $result->ticket_code ];
                }

                $ticket_codes_availability = self::sync_ticket_codes_availability( $ticket_codes );
            }

            wp_send_json( [ 'success' => $success, 'complete' => $complete , 'page' => ( $page + 1 ), 'error' => $error, 'ticket_codes' => $ticket_codes_availability ] );
        }

        /**
         * Scan through Ticket Instances and update pre-defined ticket codes statuses accordingly.
         *
         * @param $new_ticket_codes
         * @return array
         */
        function sync_ticket_codes_availability( $new_ticket_codes ) {

            $used_ticket_codes = []; $available_ticket_codes = []; // Both of these variables are being used. Please don't remove.

            if ( $new_ticket_codes ) {

                $db = new TC_PTC_DB();

                // Restructure $new_ticket_codes array
                $ticket_codes = [];
                foreach ( $new_ticket_codes as $ticket_code_arr ) {
                    $ticket_codes[] = reset( $ticket_code_arr );
                }

                // Collect Ticket Instances associated with a pre-defined ticket codes.
                $ticket_instances_ids = [];
                if ( $ticket_codes ) {
                    $ticket_instances_ids = get_posts( [
                        'post_type' => 'tc_tickets_instances',
                        'post_status' => [ 'publish', 'trash', 'draft' ],
                        'meta_query' => [ [ 'key' => 'ticket_code', 'value' => $ticket_codes, 'compare' => 'IN' ] ],
                        'fields' => 'ids',
                        'posts_per_page' => -1,
                    ]);
                }

                // Retrieve ticket_code from ticket instances meta
                foreach ( $ticket_instances_ids as $ticket_instance_id ) {
                    $ticket_code = get_post_meta( $ticket_instance_id, 'ticket_code', true );

                    if ( $ticket_code ) {
                        $used_ticket_codes[] = $ticket_code;
                    }
                }

                // Update recently imported ticket codes with "used" status.
                if ( $used_ticket_codes ) {
                    $db->update( [ 'status' => 0 ], [ 'ticket_code' => $used_ticket_codes ] );
                }

                // Update recently imported ticket codes with "available" status.
                $available_ticket_codes = array_values( array_diff( $ticket_codes, $used_ticket_codes ) );
                if ( $available_ticket_codes ) {
                    $db->update( [ 'status' => 1 ], [ 'ticket_code' => $available_ticket_codes ] );
                }
            }

            return [ 'used' => $used_ticket_codes, 'available' => $available_ticket_codes ];
        }

        /**
         * Stop Ticket Sales
         * Hide Ticket Types if there's no available pre-defined ticket codes.
         *
         * @param $is_sales_available
         * @param $ticket_type_id
         * @return bool
         */
        function set_ticket_types_unavailable( $is_sales_available, $ticket_type_id ) {

            $db = new TC_PTC_DB();
            $result = $db->select( [ 'status' => 1 ], 1 );

            if ( ! $result ) {
                $is_sales_available = false;
            }

            return $is_sales_available;
        }

        /**
         * Stop Ticket Sales
         * Restricts from proceeding to checkout if in cart items exceeds the available pre-defined ticket codes.
         *
         * @param $cart_error_number
         * @return mixed
         */
        function additional_cart_errors( $cart_error_number ) {

            global $tc;
            $cart_content_count = array_sum( $tc->get_cart_cookie() );

            $db = new TC_PTC_DB();
            $result = $db->select( [ 'status' => 1 ], $cart_content_count );

            if ( ! $result || ( count( $result ) != $cart_content_count ) ) {
                $cart_error_number++;
                tc_session_start();
                $_SESSION[ 'tc_cart_errors' ] = '<li>' . __( 'Not enough tickets left. Please update your cart with a maximum of ' . count( $result ) . ' tickets.' , 'tc' ) . '</li>';
            }

            return $cart_error_number;
        }

        /**
         * Stop Ticket Sales
         * Restricts from proceeding to checkout if in cart items exceeds the available pre-defined ticket codes.
         */
        function additional_woo_cart_errors() {

            if( is_cart() || is_checkout() ) {

                $total_quantity = 0;
                foreach( WC()->cart->get_cart() as $key => $val ) {

                    /**
                     * Use this hook to skip cart item loop.
                     */
                    if ( ! apply_filters( 'tc_skip_cart_item_loop', false, $val ) ) {
                        $total_quantity += (int) $val['quantity'];
                    }
                }

                $db = new TC_PTC_DB();
                $result = $db->select( [ 'status' => 1 ], $total_quantity );

                if ( ! $result || ( count( $result ) != $total_quantity ) ) {
                    wc_add_notice( __( 'Not enough tickets left. Please update your cart with a maximum of ' . count( $result ) . ' tickets.' , 'tc' ), 'error' );
                }
            }
        }

        /**
         * Return to settings page.
         */
        function return_to_settings_page( $additional_arguments = [] ) {

            $arguments = [
                'post_type' => 'tc_events',
                'page' => 'tc_settings',
                'tab' => 'tickera_predefined_ticket_codes',
            ];

            if ( $additional_arguments ) {
                $arguments = array_merge( $arguments, $additional_arguments );
            }

            // Redirect back to settings page
            $tab_url = add_query_arg( $arguments, admin_url('edit.php') );

            wp_redirect( $tab_url );
            exit;
        }

        /**
         * Validate ticket code if already used.
         *
         * @param $ticket_code
         * @return bool
         */
        function validate_ticket_code( $ticket_code ) {

            $ticket = get_posts( [
                'post_type' => 'tc_tickets_instances',
                'meta_key' => 'ticket_code',
                'meta_value' => $ticket_code,
                'fields' => 'ids',
                'posts_per_page' => -1
            ]);

            return ( $ticket ) ? false : true;
        }

        function sort_table() {
            $data = isset( $_POST[ 'postData' ] ) ? tickera_sanitize_array( $_POST[ 'postData' ] ) : [];
            $data[ 'order' ] = ( 'asc' == $data[ 'order' ] ) ? 'desc' : 'asc';
            $data[ 'success' ] = true;
            wp_send_json( $data );
        }
    }

    $TC_Predefined_Ticket_Codes = new TC_Predefined_Ticket_Codes();
}
