<?php

if ( ! defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

if ( ! class_exists( 'TC_PTC_DB' ) ) {

    class TC_PTC_DB {

        private $table = 'tickera_predefined_ticket_codes';

        function create_tables() {
            global $wpdb;
            require_once ABSPATH . 'wp-admin/includes/upgrade.php';
            $collate = $wpdb->has_cap( 'collation' ) ? $wpdb->get_charset_collate() : '';
            $create_query = 'CREATE TABLE ' . $wpdb->prefix . $this->table . ' (
                id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT, 
                ticket_code VARCHAR(255) NULL, 
                status TINYINT NULL DEFAULT 1,
                PRIMARY KEY (id),
                UNIQUE KEY ticket_code (ticket_code)
            ) ' . $collate;
            maybe_create_table( $wpdb->prefix . 'tickera_predefined_ticket_codes', $create_query );
        }

        /**
         * Collect Ticket Codes
         *
         * @param array $where
         * @param string $per_page
         * @param string $page
         * @param string $operator
         * @return array|object|null
         */
        function select( $where = [], $per_page = '', $page = '', $operator = '=', $sort = 'status', $sort_by = 'desc' ) {

            global $wpdb;

            // Where Clause
            $index = 0;
            $where_clause = '';
            foreach ( $where as $column => $value ) {

                if ( is_array( $value ) ) {
                    $where_clause .= ' ' . $column . ' IN (' . "'" . implode( "','", $value ) . "'" . ')';

                } else {

                    if ( 'LIKE' == $operator ) {
                        $where_clause .= ' ' . $column . ' LIKE \'%' . ( ( is_int( $value ) ) ? (int) $value : $value ) . '%\'';

                    } else {
                        $where_clause .= ' ' . $column . '=' . ( ( is_int( $value ) ) ? (int) $value : "'$value'" );
                    }
                }

                if ( ( $index + 1 ) != count( $where ) ) $where_clause .= " AND";
                $index++;
            }
            $where_clause = ( $where_clause ) ? 'WHERE' . $where_clause : '';

            // Order By Status
            $order_by = "ORDER BY {$sort} {$sort_by}";

            // Limit Clause
            $limit_clause = ( $per_page ) ? 'LIMIT ' . $per_page : '';

            // Offset Clause
            $offset = ( $per_page && $page ) ? 'OFFSET ' . ( ( $page - 1 ) * $per_page ) : '';

            $query = $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}{$this->table} %1s %1s %1s %1s",
                $where_clause,
                $order_by,
                $limit_clause,
                $offset
            );

            $result = $wpdb->get_results( stripslashes( $query ) );
            return ( $wpdb->last_error ) ? [ 'error' => $wpdb->last_error ] : $result;
        }

        /**
         * Insert new ticket codes
         *
         * @param $values array
         * @return array
         */
        function insert( $values = [] ) {

            global $wpdb;

            $query_values = '';
            foreach ( $values as $key => $value ) {
                $ticket_code = '"' . sanitize_text_field( $value[0] ) . '"';
                $query_values .= '(' . $ticket_code . ')';

                // Insert comma trailing if value is not the last element of $values.
                if ( ( $key + 1 ) != count( $values ) ) {
                    $query_values .= ' ,';
                }
            }

            $query = "INSERT INTO {$wpdb->prefix}{$this->table} (ticket_code) VALUES {$query_values}";
            $result = $wpdb->query( stripslashes( $query ) );

            return ( $wpdb->last_error ) ? [ 'error' => $wpdb->last_error ] : [ 'success' => $result ];
        }

        /**
         * Update Existing Ticket Codes
         *
         * @param array $values
         * @param $where
         * @return array
         */
        function update( $values = [], $where = [] ) {

            global $wpdb;

            if ( ! $values ) {
                return [ 'error' => __( 'No supplied values to update.', 'tc' ) ];
            }

            // Set Clause
            $index = 0;
            $set_clause = '';
            foreach ( $values as $column => $value ) {
                $column = sanitize_key( $column );
                $value = ( is_int( $value ) ) ? (int) $value : "'" . sanitize_text_field( $value ) . "'";
                $set_clause .=  $column . '=' . $value;

                if ( ( $index + 1 ) != count( $values ) ) {
                    $set_clause .= ',';
                }
                $index++;
            }
            $set_clause = ( $set_clause ) ? 'SET ' . $set_clause : '';

            // Where Clause
            $index = 0;
            $where_clause = 'WHERE ';
            foreach ( $where as $column => $value ) {

                if ( is_array( $value ) ) {
                    $where_clause .= $column . " IN ('" . implode( "','", $value ) . "')";

                } else {
                    $where_clause .= $column . '=' . ( is_int( $value ) ? (int) $value : "'" . sanitize_text_field( $value ) . "'" ) ;
                }

                if ( ( $index + 1 ) != count( $where ) ) {
                    $where_clause .= ' AND ';
                }
                $index++;
            }

            $query = $wpdb->prepare( "UPDATE {$wpdb->prefix}{$this->table} %1s %1s", $set_clause, $where_clause );
            $result = $wpdb->query( stripslashes( $query ) );

            return ( $wpdb->last_error ) ? [ 'error' => $wpdb->last_error ] : [ 'success' => $result ];
        }

        /**
         * Delete Ticket Codes by ID
         *
         * @param array $where
         * @return array
         */
        function delete( $where = [] ) {
            global $wpdb;

            // Where Clause
            $index = 0;
            $where_clause = 'WHERE ';
            foreach ( $where as $column => $value ) {
                $where_clause .= $column . '=' . ( is_int( $value ) ? (int) $value : "'" . sanitize_text_field( $value ) . "'" ) ;

                if ( ( $index + 1 ) != count( $where ) ) {
                    $where_clause .= ' AND ';
                }
                $index++;
            }

            $query = $wpdb->prepare( "DELETE FROM {$wpdb->prefix}{$this->table} %1s", $where_clause );
            $result = $wpdb->query( stripslashes( $query ) );
            return ( $wpdb->last_error ) ? [ 'error' => $wpdb->last_error ] : [ 'success' => $result ];
        }

        function truncate() {

            global $wpdb;
            $result = $wpdb->query( $wpdb->prepare( "DROP TABLE {$wpdb->prefix}%1s", $this->table ) );

            if ( $wpdb->last_error ) {
                return [ 'error' => $wpdb->last_error ];

            } else {
                self::create_tables();
                return [ 'success' => $result ];
            }
        }
    }
}
